<html>
<head>
    <link href="https://unpkg.com/tailwindcss@^2/dist/tailwind.min.css" rel="stylesheet">
</head>
<body>
<div id="app" class="p-5">
    <h1 class="text-3xl font-bold text-yellow-800 mb-5">ウェブでレジ入力機能・サンプル</h1>
    <div class="my-5" v-if="!focusing">
        <span class="bg-red-500 text-red-50 rounded p-3 font-bold">&#x26A0; JANコードを入力できません。画面にフォーカスしてください。</span>
    </div>
    <table class="borer border-2 border-gray-200">
        <thead class="bg-blue-200">
            <tr>
                <th class="p-2">商品名</th>
                <th class="p-2">価格</th>
                <th class="p-2">数量</th>
                <th class="p-2">数量変更</th>
                <th class="p-2">削除</th>
            </tr>
        </thead>
        <tbody class="bg-gray-50">
            <!-- ① 読み込まれた商品の一覧を表示する -->
            <tr v-for="item in items">
                <td class="p-2" v-text="item.title"></td>
                <td class="p-2" v-text="item.price"></td>
                <td class="p-2 text-center" v-text="item.quantity"></td>
                <td class="p-2 text-center">
                    <button class="bg-blue-500 px-2 py-1 text-blue-50 rounded ml-4" type="button" @click="onEdit(item)">変更</button>
                </td>
                <td class="p-2 text-center">
                    <button class="bg-red-500 px-2 py-1 text-red-50 rounded" type="button" @click="onDelete(item)">削除</button>
                </td>
            </tr>
            <!-- ② 価格の合計金額を表示する -->
            <tr>
                <td class="p-3 font-bold text-3xl text-right" colspan="5">
                    <span v-text="totalPrice"></span>円
                    <span class="text-sm font-normal text-gray-500">（税: <span v-text="totalTax"></span>円）</span>
                </td>
            </tr>
        </tbody>
    </table>
    <!-- ③ JANコードの入力ボックス -->
    <input
        ref="janCodeInput"
        style="position:absolute;top:-100px;left:-100px;"
        type="text"
        v-model="janCode"
        @input="onInput"
        @focus="onFocus"
        @blur="onBlur">
</div>
<script src="https://unpkg.com/vue@3.0.2/dist/vue.global.prod.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/axios/0.19.2/axios.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/lodash.js/4.17.20/lodash.min.js"></script>
<script>

    Vue.createApp({
        data() {
            return {
                focusing: false, // <input> が入力状態かどうか
                submitting: false, // Ajax送信中かどうか
                janCode: '',
                items: [], // 読み取り済みのJANコード
            }
        },
        methods: {
            getItem(janCode) {  // ④ Ajaxで商品データを取得する

                this.submitting = true; // 「送信中」に変更
                const url = `/cash_register/${janCode}`;
                axios.get(url)
                    .then(response => {

                        if(response.data.result === true) {

                            const item = response.data.item;
                            this.addItem(item);

                        } else {

                            alert('この商品は存在しません...');

                        }

                    })
                    .catch(error => {

                        alert('予期しないエラーが発生しました...');

                    })
                    .finally(() => { // 成功しようが失敗しようが必ず通る場所

                        this.submitting = false; // 送信状態を初期化
                        this.janCode = ''; // JANコードを初期化

                    });

            },
            onInput() { // ⑤ 読み取ったJANコードが入力されたときに実行

                if(this.submitting === false) {

                    const janCode = this.emToEn(this.janCode);

                    if(janCode.match(/^[0-9]{13}$/)) { // 半角数字が13文字の場合だけ実行する

                        this.getItem(janCode);

                    }

                }

            },
            onFocus() { // ⑥-1 入力ボックスが入力状態になったとき

                this.focusing = true;

            },
            onBlur() { // ⑥-2 入力ボックスが入力状態ではなくなったとき

                this.focusing = false;
                this.janCodeInput.focus(); // すぐ入力状態へ戻す

            },
            onDelete(item) { // ⑦ 削除するボタンがクリックされたとき

                const message = `『${item.title}』を削除します。よろしいですか？`;

                if(confirm(message)) {

                    _.remove(this.items, {
                        jan_code: item.jan_code
                    })

                }

            },
            onEdit(item) { // ⑧ 個数の変更ボタンがクリックされたとき

                let newQuantity = prompt('件数を変更してください。', item.quantity);
                newQuantity = parseInt(
                    this.emToEn(newQuantity)
                );

                if(newQuantity > 0) {

                    item.quantity = newQuantity;

                }

            },
            addItem(addingItem) {

                const item = _.find(this.items, { // 同じJANコードの商品を探す
                   jan_code: addingItem.jan_code
                });

                if(item !== undefined) { // すでに同じ商品が読み取られている場合

                    item.quantity++;

                } else { // 新しく読み取られた場合

                    addingItem['quantity'] = 1;
                    this.items.push(addingItem);

                }

            },
            emToEn(str) { // 全角数字を半角へ変換

                return str.replace(/[０-９]/g, s => {

                    return String.fromCharCode(s.charCodeAt(0) - 0xFEE0);

                });

            }
        },
        computed: {
            // ⑨ 合計金額を計算する
            totalPrice() {

                return _.sumBy(this.items, item => {

                    return item.price * item.quantity;

                }) + this.totalTax;

            },
            totalTax() {

                return _.sumBy(this.items, item => {

                    return Math.floor(item.price * item.tax_percent * item.quantity * 0.01);

                });

            }
        },
        mounted() {

            this.janCodeInput.focus(); // <input> を入力状態にする

            // テストのちょっとしたTip
            // this.getItem('9784534051943');
            // this.getItem('9784839928407');

        },
        setup() {

            return {
                janCodeInput: Vue.ref(null) // ⑩ <input ref="janCodeInput"...> にアクセスできるようにする
            }

        }
    }).mount('#app');

</script>
</body>
</html>
